//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.12

import wesual.Ui 1.0

import wesual.Ui.Private 1.0

/*!
\qmltype     UiRadioButton
\inqmlmodule we.captivo.Ui
\since       we.captivo.Ui 1.0
\inherits    UiButtonBase
\ingroup     captivo-ui-controls

\brief A radio button with a text label.

The UiRadioButton component provides a button that can be grouped together
to form a single choice selection:

\qml
import QtQuick 2.4

import wesual.Ui 1.0

Column {
    spacing : 10

    UiRadioButtonGroup {
        id : buttonGroup
        activeButton : option1
    }

    UiRadioButton {
        id : option1
        text  : "Option 1"
        group : buttonGroup
    }
    UiRadioButton {
        id : option2
        text  : "Option 2"
        group : buttonGroup
    }
    UiRadioButton {
        id : option3
        text  : "Option 3"
        group : buttonGroup
    }
}
\endqml

\note UiRadioButtonColumn provides a more convenient and visually consistent
way to use radio buttons.

\sa UiRadioButtonGroup UiRadioButtonColumn
*/

UiButtonBase {
    id : uiRadioButton

    /*!
    \qmlproperty string UiRadioButton::text
    \brief Defines the label text displayed beside the radio button.

    \elideright
    */
    property alias text : label.text

    /*!
    \brief The radio button group this button belongs to.

    The UiRadioButtonGroup links several UiRadioButton controls so that only
    only one button can be checked at a time.

    \note A UiRadioButton without an assigned group is not functional and cannot
    be checked.
    */
    property UiRadioButtonGroup group : null

    /*!
    \brief Defines the checked state of the control where a value of \c true
    indicates that the box is currently checked.

    \note The checked state is controlled by the assigned \l group and cannot
    be modified directly. To explicitly check a radio button set the
    \l {UiRadioButtonGroup::}{activeButton} of the corresponding group.
    */
    readonly property bool checked :
        group && group.activeButton === uiRadioButton

    property bool bindable : false

    property UiRadioButtonVisualStyle visualStyle : null

    implicitHeight : p_.size
    implicitWidth  : p_.size + p_.spacing + label.implicitWidth + p_.margin
    baselineOffset : label.y + label.baselineOffset

    onClicked : {
        if (group && !bindable) {
            group.activeButton = uiRadioButton;
        }
    }

    QtObject {
        id : p_

        readonly property int size          : 18
        readonly property int checkMarkSize : 6
        readonly property int spacing       : 6
        readonly property int margin        : 2

        function check() {
            if (uiRadioButton.group) {
                uiRadioButton.group.activeButton = uiRadioButton;
            }
        }
    }

    Rectangle {
        id : box

        width  : p_.size
        height : width
        radius : width / 2.0
        color  : UiColors.getColor(UiColors.White)
        border {
            width : 1
            color : UiColors.getColor(UiColors.DarkerControlGrey22)
        }

        states : [
            State {
                name : "checked"
                when : uiRadioButton.checked

                PropertyChanges {
                    target  : checkMark
                    opacity : 1
                }
            }
        ]

        Rectangle {
            id : checkMark

            anchors.centerIn : parent

            width   : p_.checkMarkSize
            height  : p_.checkMarkSize
            radius  : width / 2
            opacity : 0
            color   : UiColors.getColor(UiColors.MediumDarkGrey67)
        }
    }

    UiText {
        id : label

        visible  : text.length > 0
        width    : uiRadioButton.width - p_.size - p_.spacing - p_.margin
        anchors {
            left           : box.right
            leftMargin     : p_.spacing
            verticalCenter : uiRadioButton.verticalCenter
        }
    }

    UiFocusRect {
        anchors.centerIn : label
        width            : label.width + p_.margin * 2
        height           : label.height + 1
        target           : uiRadioButton
    }

    states: [
        State {
            name : "disabled"
            when : !uiRadioButton.enabled

            PropertyChanges {
                target : checkMark
                color  : UiColors.getColor(UiColors.DisabledLightGrey13)
            }
            PropertyChanges {
                target       : box
                border.color : UiColors.getColor(UiColors.DisabledLightGrey13)
            }
        },
        State {
            name : "clicked"
            when : uiRadioButton.pressed

            PropertyChanges {
                target       : box
                color        : uiRadioButton.visualStyle ?
                                   uiRadioButton.visualStyle.selectionColor :
                                   UiColors.getColor(UiColors.SelectionGreen)
                border.color : uiRadioButton.visualStyle ?
                                   uiRadioButton.visualStyle.
                                   selectionBorderColor :
                                   UiColors.getColor(
                                       UiColors.DarkerSelectionGreen)
            }
        },
        State {
            name : "hovered"
            when : uiRadioButton.hovered || uiRadioButton.activeFocus
            changes : [
                PropertyChanges {
                    target       : box
                    color        : uiRadioButton.visualStyle ?
                                       uiRadioButton.visualStyle.hoverColor :
                                       UiColors.getColor(UiColors.HoverGreen)
                    border.color : uiRadioButton.visualStyle ?
                                       uiRadioButton.visualStyle.
                                       hoverBorderColor :
                                       UiColors.getColor(
                                           UiColors.DarkerHoverGreen)
                }
            ]
        }
    ]
}
